!=======================================================================
!# Database mp(k,q) of mass flow rates using Chebyshev interpolation. 
!# Transformation F(q)= log(-mp); mp~=-exp(F~(q)) is used irrespective of e.

!# Notation: 
!   c: dimensionless curvature ;   e: eccentricity epsilon
!  dk: Knudsen number k        ;  xx:=chi/(2*pi) (0<xx<1)
!   q:= |2*xx-1|
! fmp:= m_Pec                  ; fmc:= m_Cec
! fma:= S_Pec at y=0           ; fmb:= S_Cec at y=0
!  nx:= Nm                     ; ny:= Nn

!# Usage and notes:
!  See the main program to learn how to use this program. 
!  Please remove the main program and use the following subroutine as a library. 
!  This program is for (c,epsilon,Ta,Tb)=(0.025,0.5,2,1) epsilon=0.5, Ta=2 and Tb=1. 
!  For other (c,eplison,Ta,Tb), e.g., (c,epsilon,Ta,Tb)=(0.05,0.5,1,2)
! please replace the data statement below and dka=... and so on 
! by those of "0550t12.dat", etc. For the parameter-filename correspondance, see 00readme.txt
!  The first line of 0510.dat reads: 
!    Nm, Nn, dka, dkb, ya, yb, c, epsilon, ..., 
!-----------------------------------------------------------------------

!{ a sample main program to get the data drawing 2D plot of m_P(dk,xx)
program main 
implicit real*8(a-h,o-z)
pi= 3.14159265358979324d0
sqrtpi= 1.77245385090552 
pi2= pi*2
c= 0.025
e= 0.5
rk= (100.0)**(1.0/10)
dy= 0.5d0/10
do  j= 0,10
do  i= 0,10
  dk= 0.1*rk**i  ! dk runs over 0.1--10
  xx= 0.0 +dy*j  ! xx runs over 0.0--0.5
  q= abs(2*xx -1)
  call mps0250(fmp,fmc,fma,fmb,dk,q,c,e)
  write(*,'(9e13.5)') dk,2*xx,-fmp,fmc,fma,fmb
enddo
write(*,*)' '
enddo
write(*,'(a4,9f8.4)') '#ce=',c,e
end
!}

!=======================================================================
subroutine mps0250(fmp,fmc,fma,fmb,dk,q,c,e)
!-----------------------------------------------------------------------
implicit real*8(a-h,o-z)
parameter(mx=8-1,my=7-1,mpc=4)  !NOTE: mx must be nx-1 and my must be ny-1
dimension tnx(0:mx),tny(0:my),cc(mpc,0:mx,0:my)

!{ Chebyshev coefficients
nx=   8
ny=   7
if(nx/=mx+1) stop
if(ny/=my+1) stop
dka= 0.5000E-01 
dkb= 0.2000E+02 
ya = 0.0000E+00 
yb = 0.1000E+01 
c =  0.5000D-01 
e =  0.5000D+00 
ta= 2
tb= 1
nny= 160
!   8   7  0.5000E-01  0.2000E+02  0.0000E+00  0.1000E+01  0.2500D-01  0.5000D+00 160
! mp             mc             sp             sc
data cc/ &
 -0.373612E+00, -0.119152E+01, -0.851770E+00, -0.122321E+01, &
 -0.146699E+00, -0.211955E+00, -0.145794E+00,  0.924045E+00, &
  0.247329E+00,  0.843942E-02,  0.920209E-03, -0.349150E+00, &
 -0.938126E-01,  0.109747E-01,  0.681954E-02,  0.557415E-01, &
  0.267095E-02,  0.279185E-03,  0.345418E-03,  0.705004E-02, &
  0.495520E-02,  0.299396E-03,  0.467044E-03, -0.412575E-02, &
 -0.399283E-03, -0.298962E-03, -0.507554E-04,  0.780638E-04, &
  0.843835E-04, -0.379048E-03, -0.313612E-03,  0.280929E-03, &
  0.676929E+00,  0.624572E+00, -0.366510E-03, -0.211849E+00, &
 -0.242163E+00, -0.412848E-01, -0.267875E-01,  0.270678E+00, &
  0.969927E-01, -0.148892E-01, -0.120192E-01, -0.639295E-01, &
 -0.526726E-02,  0.555723E-02,  0.308973E-02, -0.117833E-01, &
 -0.864733E-02,  0.902633E-03,  0.106058E-02,  0.916662E-02, &
  0.251860E-02, -0.421246E-03, -0.452604E-03, -0.154087E-03, &
 -0.130231E-03,  0.484776E-03,  0.262249E-03, -0.121663E-02, &
 -0.150163E-03, -0.338237E-03, -0.409923E-04,  0.494382E-03, &
  0.882022E-01,  0.720730E-01, -0.350344E-03, -0.324745E-01, &
 -0.408291E-01, -0.503834E-02, -0.309097E-02,  0.381429E-01, &
  0.115257E-01, -0.322170E-02, -0.236556E-02, -0.502289E-02, &
  0.189852E-02,  0.465565E-03,  0.216517E-04, -0.369877E-02, &
 -0.195942E-02,  0.550475E-03,  0.497928E-03,  0.101465E-02, &
  0.327851E-03, -0.257898E-03, -0.138844E-03,  0.599275E-03, &
  0.115409E-03,  0.125226E-03,  0.388107E-06, -0.427865E-03, &
 -0.118431E-03, -0.192234E-04,  0.349562E-04,  0.121730E-03, &
 -0.670112E-01, -0.643057E-01,  0.163702E-04,  0.198779E-01, &
  0.217714E-01,  0.415584E-02,  0.267589E-02, -0.264240E-01, &
 -0.102982E-01,  0.120262E-02,  0.105456E-02,  0.735213E-02, &
  0.136777E-02, -0.725229E-03, -0.456129E-03,  0.727785E-03, &
  0.646772E-03,  0.764989E-04, -0.144675E-04, -0.112999E-02, &
 -0.304760E-03, -0.153610E-04,  0.403713E-04,  0.254843E-03, &
  0.886597E-04, -0.439870E-04, -0.375388E-04,  0.431593E-04, &
 -0.241218E-04,  0.504618E-04,  0.882735E-05, -0.421618E-04, &
 -0.245953E-01, -0.202732E-01,  0.256458E-04,  0.894346E-02, &
  0.114191E-01,  0.138452E-02,  0.976582E-03, -0.106366E-01, &
 -0.342248E-02,  0.897808E-03,  0.623440E-03,  0.151745E-02, &
 -0.473093E-03, -0.180573E-03, -0.712652E-04,  0.104212E-02, &
  0.593905E-03, -0.122657E-03, -0.904058E-04, -0.365490E-03, &
 -0.121019E-03,  0.656013E-04,  0.338732E-04, -0.118793E-03, &
 -0.377863E-04, -0.388036E-04, -0.145153E-04,  0.111226E-03, &
  0.457875E-04,  0.119847E-04,  0.125840E-05, -0.371660E-04, &
 -0.459170E-02, -0.308607E-02,  0.348095E-04,  0.194091E-02, &
  0.260113E-02,  0.242367E-03,  0.121474E-03, -0.201049E-02, &
 -0.329377E-03,  0.229075E-03,  0.186588E-03, -0.441322E-04, &
 -0.352873E-03,  0.332013E-04,  0.297751E-04,  0.299467E-03, &
  0.147713E-03, -0.785633E-04, -0.544599E-04,  0.271882E-04, &
  0.153080E-04,  0.309380E-04,  0.162563E-04, -0.106713E-03, &
 -0.334174E-04, -0.783911E-05, -0.167880E-05,  0.461521E-04, &
  0.152319E-04, -0.748422E-05, -0.223341E-05, -0.831305E-05, &
  0.404993E-03,  0.497302E-03, -0.353316E-04, -0.831025E-04, &
  0.181072E-04, -0.200593E-04,  0.637848E-04,  0.200990E-03, &
  0.159231E-03, -0.625016E-05, -0.615992E-04, -0.123492E-03, &
 -0.782895E-04,  0.431019E-04,  0.322128E-04, -0.191279E-04, &
  0.918598E-05, -0.227500E-04,  0.932116E-05,  0.652319E-04, &
  0.113843E-05,  0.341212E-05, -0.189921E-04, -0.335766E-04, &
 -0.108937E-04,  0.396235E-05,  0.563833E-05,  0.361711E-05, &
  0.155479E-04, -0.395571E-05,  0.451193E-05,  0.368411E-05  &
/
!}

!{ interpolation
za= log(dka)
zb= log(dkb)
wa= ya
wb= yb
x= 2*(log(dk)-(zb+za)*0.5)/(zb-za)
y= 2*q -1d0
fmp= 0.0
fmc= 0.0
fma= 0.0
fmb= 0.0
tnx(0)= 1.0
tnx(1)= x
tny(0)= 1.0
tny(1)= y
do  k= 2,mx
  tnx(k)= tnx(k-1)*2*x -tnx(k-2)
enddo
do  l= 2,my
  tny(l)= tny(l-1)*2*y -tny(l-2)
enddo
do  k= 0,nx-1
do  l= 0,ny-1
  fmp= fmp+ cc(1,k,l)*tnx(k)*tny(l)
  fmc= fmc+ cc(2,k,l)*tnx(k)*tny(l)
  fma= fma+ cc(3,k,l)*tnx(k)*tny(l)
  fmb= fmb+ cc(4,k,l)*tnx(k)*tny(l)
enddo
enddo
fmp=-exp(fmp)
fmc= exp(fmc)
fma= exp(fma)
fmb= exp(fmb)
!}
end
